import { create } from 'zustand'

export type AppId = 'about' | 'python' | 'projects' | 'presence' | 'skills' | 'contact'

export type WinState = {
  id: AppId
  title: string
  minimized: boolean
  z: number
  x: number
  y: number
  w: number
  h: number
  maximized?: boolean
  prev?: { x: number; y: number; w: number; h: number } | null
}

type Store = {
  windows: Partial<Record<AppId, WinState>>
  zTop: number
  open: (id: AppId, init?: Partial<WinState>) => void
  focus: (id: AppId) => void
  minimize: (id: AppId) => void
  close: (id: AppId) => void
  move: (id: AppId, pos: { x: number; y: number }) => void
  resize: (id: AppId, size: { w: number; h: number }) => void
  toggleMaximize: (id: AppId) => void
}

export const useWindowStore = create<Store>((set, get) => ({
  windows: {},
  zTop: 10,
  open: (id, init) =>
    set((s) => {
      const z = s.zTop + 1
      const titleMap: Record<AppId, string> = {
        about: 'About',
        python: 'Python Editor',
        projects: 'Projects',
        presence: 'Discord',
        skills: 'Skills',
        contact: 'Contact',
      }
      return {
        zTop: z,
        windows: {
          ...s.windows,
          [id]: {
            id,
            title: titleMap[id],
            minimized: false,
            z,
            x: init?.x ?? 160,
            y: init?.y ?? 120,
            w: init?.w ?? 760,
            h: init?.h ?? 520,
            ...init,
          },
        },
      }
    }),
  focus: (id) =>
    set((s) => {
      const win = s.windows[id]
      if (!win) return {}
      const z = s.zTop + 1
      return { zTop: z, windows: { ...s.windows, [id]: { ...win, minimized: false, z } } }
    }),
  minimize: (id) =>
    set((s) => {
      const win = s.windows[id]
      if (!win) return {}
      return { windows: { ...s.windows, [id]: { ...win, minimized: true } } }
    }),
  close: (id) =>
    set((s) => {
      const w = { ...s.windows }
      delete w[id]
      return { windows: w }
    }),
  move: (id, pos) =>
    set((s) => {
      const win = s.windows[id]
      if (!win) return {}
      const x = Math.max(0, pos.x)
      const y = Math.max(28, pos.y) // keep below menu bar
      return { windows: { ...s.windows, [id]: { ...win, x, y, maximized: false } } }
    }),
  resize: (id, size) =>
    set((s) => {
      const win = s.windows[id]
      if (!win) return {}
      const w = Math.max(360, size.w)
      const h = Math.max(240, size.h)
      return { windows: { ...s.windows, [id]: { ...win, w, h, maximized: false } } }
    }),
  toggleMaximize: (id) =>
    set((s) => {
      const win = s.windows[id]
      if (!win) return {}
      if (!win.maximized) {
        const margin = 12
        const top = 28
        const bottomSafe = 88 // leave room for dock
        const availW = Math.max(360, window.innerWidth - margin * 2)
        const availH = Math.max(240, window.innerHeight - top - bottomSafe - margin)
        return {
          windows: {
            ...s.windows,
            [id]: {
              ...win,
              prev: { x: win.x, y: win.y, w: win.w, h: win.h },
              x: margin,
              y: top + margin,
              w: availW,
              h: availH,
              maximized: true,
            },
          },
        }
      } else {
        const prev = win.prev
        if (prev) {
          return { windows: { ...s.windows, [id]: { ...win, ...prev, maximized: false, prev: null } } }
        }
        return { windows: { ...s.windows, [id]: { ...win, maximized: false } } }
      }
    }),
}))
