"use client";
// Lightweight Pyodide loader for client-side Python execution.
// Loads Pyodide from CDN once and exposes helpers to run code with stdout/stderr capture.

let _pyodide: any | null = null;
let _loading: Promise<any> | null = null;

export async function getPyodide() {
  if (_pyodide) return _pyodide;
  if (_loading) return _loading;
  _loading = (async () => {
    // Inject script if needed
    const existing = (globalThis as any).loadPyodide;
    if (!existing) {
      await new Promise<void>((resolve, reject) => {
        const s = document.createElement("script");
        s.src = "https://cdn.jsdelivr.net/pyodide/v0.25.1/full/pyodide.js";
        s.async = true;
        s.onload = () => resolve();
        s.onerror = () => reject(new Error("Failed to load Pyodide script"));
        document.head.appendChild(s);
      });
    }
    const loadPyodide = (globalThis as any).loadPyodide as (opts: any) => Promise<any>;
    const pyodide = await loadPyodide({ indexURL: "https://cdn.jsdelivr.net/pyodide/v0.25.1/full/" });
    _pyodide = pyodide;
    return pyodide;
  })();
  return _loading;
}

export type Stdio = {
  stdout?: (text: string) => void;
  stderr?: (text: string) => void;
};

export async function runPython(code: string, io: Stdio = {}) {
  const py = await getPyodide();
  const restore: { out?: any; err?: any } = {};
  try {
    if (io.stdout) {
      restore.out = py.setStdout({ batched: (s: string) => io.stdout!(s) });
    }
    if (io.stderr) {
      restore.err = py.setStderr({ batched: (s: string) => io.stderr!(s) });
    }
    // Use runPythonAsync so long-running code yields to the event loop
    return await py.runPythonAsync(code);
  } finally {
    // Restore stdio if provided
    if (restore.out) restore.out();
    if (restore.err) restore.err();
  }
}
