"use client";
import * as React from "react";
import clsx from "clsx";
import { getPyodide, runPython } from "@/lib/pyodideClient";

// Tiny client-side Python syntax highlighter (regex-based)
function highlightPython(src: string): string {
  // Escape HTML first
  const esc = src
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;");
  // Strings (single, double, triple)
  let out = esc
    .replace(/([rubf]*)("""[\s\S]*?"""|'''[\s\S]*?'''|"[^"\\\n]*(?:\\.[^"\\\n]*)*"|'[^'\\\n]*(?:\\.[^'\\\n]*)*')/gi, (_m, p1, p2) => `<span class="tok-str">${p1}${p2}</span>`)
    // Comments
    .replace(/(^|\s)#([^\n]*)/g, (_m, p1, p2) => `${p1}<span class="tok-cmt">#${p2}</span>`)
    // Numbers
    .replace(/\b(0x[0-9a-fA-F_]+|\d+\.\d*|\d*\.\d+|\d+)(j|J)?\b/g, '<span class="tok-num">$1$2</span>');

  // Keywords and builtins (simple pass after strings/comments)
  const kw = new RegExp(
    `\\b(${[
      'False','None','True','and','as','assert','async','await','break','class','continue','def','del','elif','else','except','finally','for','from','global','if','import','in','is','lambda','nonlocal','not','or','pass','raise','return','try','while','with','yield'
    ].join('|')})\\b`,
    'g'
  );
  out = out.replace(kw, '<span class="tok-kw">$1</span>');

  const builtins = new RegExp(
    `\\b(${[
      'print','len','range','list','dict','set','tuple','int','float','str','bool','sum','min','max','open','enumerate','map','filter','zip'
    ].join('|')})\\b`,
    'g'
  );
  out = out.replace(builtins, '<span class="tok-builtin">$1</span>');

  return out;
}

export default function PythonEditor() {
  const [code, setCode] = React.useState<string>(defaultSnippet.trim());
  const [output, setOutput] = React.useState<string>("Ready.");
  const [mounted, setMounted] = React.useState(false);
  const [loadingPy, setLoadingPy] = React.useState(false);
  const [running, setRunning] = React.useState(false);
  React.useEffect(() => setMounted(true), []);

  const appendOut = React.useCallback((text: string) => {
    setOutput((prev) => (prev ? prev + text : text));
  }, []);

  const run = async () => {
    // Lazy-load Pyodide and run code with stdout/stderr capture
    try {
      setRunning(true);
      // If not yet initialized, show a brief status and initialize first
      if (!(window as any).loadPyodide || !(await (async () => (await (getPyodide().catch(() => null))) ? true : false)())) {
        setLoadingPy(true);
        setOutput("Initializing Python runtime (Pyodide)… This may take a few seconds on first run.\n");
        await getPyodide();
        setLoadingPy(false);
      }
      setOutput("");
      await runPython(code, {
        stdout: (s) => appendOut(s),
        stderr: (s) => appendOut(s),
      });
    } catch (e: any) {
      appendOut(`\n[Error] ${e?.message || String(e)}\n`);
    } finally {
      setRunning(false);
    }
  };

  const clear = () => setOutput("");

  return (
    <div className={clsx(
      'p-3 md:p-4 text-sm h-full grid gap-3',
      'md:grid-cols-2',
      mounted ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-1',
      'transition-all duration-300 ease-out motion-reduce:transition-none'
    )}>
      <section className="min-h-0 flex flex-col">
        <header className="flex items-center justify-between gap-2 mb-2">
          <h2 className="text-base font-semibold">Python Editor</h2>
          <div className="flex items-center gap-2">
            <button disabled={running || loadingPy} onClick={run} className={clsx("rounded-md px-3 py-1.5 text-white shadow focus:outline-none focus-visible:ring-2 focus-visible:ring-accent/60", (running || loadingPy) ? "bg-accent/50 cursor-not-allowed" : "bg-accent/90 hover:shadow-md")}>{loadingPy ? 'Loading…' : (running ? 'Running…' : 'Run')}</button>
            <button onClick={clear} className="rounded-md px-3 py-1.5 bg-black/5 dark:bg-white/10 hover:bg-black/10 dark:hover:bg-white/15 text-zinc-800 dark:text-zinc-100">Clear Output</button>
          </div>
        </header>
        <div className="relative grid grid-rows-[1fr] min-h-0 flex-1">
          {/* Overlay highlighted code */}
          <pre aria-hidden className="absolute inset-0 m-0 overflow-auto rounded-xl border border-black/10 dark:border-white/10 bg-white/70 dark:bg-zinc-900/70 p-3 font-mono text-[12.5px] leading-5 whitespace-pre-wrap break-words select-none pointer-events-none">
            <code dangerouslySetInnerHTML={{ __html: highlightPython(code) }} />
          </pre>
          {/* Transparent textarea on top for editing */}
          <textarea
            value={code}
            onChange={(e) => setCode(e.target.value)}
            spellCheck={false}
            className="relative z-10 resize-none rounded-xl border border-black/10 dark:border-white/10 bg-transparent p-3 font-mono text-[12.5px] leading-5 text-transparent caret-current selection:bg-accent/20 selection:text-inherit focus:outline-none"
            style={{ WebkitTextFillColor: 'transparent' }}
          />
        </div>
        <p className="mt-2 text-xs text-zinc-500">Runs locally in your browser via Pyodide (WebAssembly). First run may download the runtime (~7–8MB).</p>
      </section>

      <section className="min-h-0 flex flex-col">
        <header className="mb-2">
          <h3 className="text-base font-semibold">Output</h3>
        </header>
        <div className="rounded-xl border border-black/10 dark:border-white/10 bg-white/70 dark:bg-zinc-900/70 p-3 font-mono text-[12.5px] leading-5 whitespace-pre-wrap break-words overflow-auto flex-1">
          {output || <span className="text-zinc-500">(empty)</span>}
        </div>
      </section>

      {/* Inline styles for tokens (scoped via component wrapper) */}
      <style>{`
        .tok-kw{ color:#7c4dff; font-weight:600 }
        .tok-builtin{ color:#0ea5e9 }
        .tok-str{ color:#10b981 }
        .tok-cmt{ color:#9ca3af; font-style:italic }
        .tok-num{ color:#f59e0b }
      `}</style>
    </div>
  );
}

const defaultSnippet = `# Python demo\n# Write your code on the left.\n\nprint('Hello from Python editor!')\nfor i in range(3):\n    print('Line', i+1)`;
