"use client";
import { AppId, useWindowStore } from "@/lib/windowStore";
import { useEffect, useMemo, useRef, useState } from "react";
import clsx from "clsx";

const apps: { id: AppId; label: string; icon: React.ReactNode }[] = [
  { id: "about", label: "About", icon: <AboutIcon /> },
  { id: "python", label: "Python Editor", icon: <PythonIcon /> },
  { id: "projects", label: "Projects", icon: <ProjectsIcon /> },
  { id: "presence", label: "Discord", icon: <DiscordIcon className="translate-y-[1px]" /> },
  { id: "skills", label: "Skills", icon: <SkillsIcon /> },
  { id: "contact", label: "Contact", icon: <MailIcon /> },
];

export default function Dock() {
  const open = useWindowStore((s) => s.open);
  const focus = useWindowStore((s) => s.focus);
  const minimize = useWindowStore((s) => s.minimize);
  const windows = useWindowStore((s) => s.windows);

  const ref = useRef<HTMLDivElement>(null);
  const [pointerX, setPointerX] = useState<number | null>(null);
  const [bouncingId, setBouncingId] = useState<AppId | null>(null);

  useEffect(() => {
    if (!ref.current) return;
    const el = ref.current;
    const onMove = (e: PointerEvent) => {
      const rect = el.getBoundingClientRect();
      if (e.clientY < rect.top - 80 || e.clientY > rect.bottom + 80) {
        setPointerX(null);
        return;
      }
      setPointerX(e.clientX);
    };
    const onLeave = () => setPointerX(null);
    window.addEventListener("pointermove", onMove);
    window.addEventListener("pointerleave", onLeave);
    return () => {
      window.removeEventListener("pointermove", onMove);
      window.removeEventListener("pointerleave", onLeave);
    };
  }, []);

  return (
    <div className="fixed inset-x-0 bottom-4 z-[900] flex justify-center">
      <div ref={ref} className="glass rounded-2xl px-3 py-2 shadow-dock border border-black/10 dark:border-white/10 flex gap-2">
        {apps.map((app, idx) => {
          const isOpen = !!windows[app.id];
          const isMinimized = windows[app.id]?.minimized;
          // Magnification calculation
          let scale = 1;
          if (pointerX != null && ref.current) {
            const item = ref.current.children[idx] as HTMLElement | undefined;
            if (item) {
              const rect = item.getBoundingClientRect();
              const center = rect.left + rect.width / 2;
              const dist = Math.abs(pointerX - center);
              const influence = Math.max(0, 1 - dist / 140); // 140px influence radius
              scale = 1 + 0.6 * Math.pow(influence, 2); // curve for smoother effect
            }
          }
          return (
            <button
              key={app.id}
              className="relative flex flex-col items-center will-change-transform motion-safe:transition-transform motion-safe:duration-150 ease-out hover:-translate-y-1 active:translate-y-0"
              style={{ transform: `scale(${scale})`, transformOrigin: "bottom" }}
              onClick={() => {
                if (!isOpen) {
                  open(app.id);
                  setBouncingId(app.id);
                  window.setTimeout(() => setBouncingId((id) => (id === app.id ? null : id)), 650);
                }
                else if (isMinimized) focus(app.id);
                else minimize(app.id);
              }}
              title={app.label}
            >
              <div className={clsx(
                "size-12 rounded-2xl overflow-hidden grid place-items-center bg-white dark:bg-zinc-900",
                "border border-black/10 dark:border-white/10 shadow-md"
              , bouncingId === app.id && "motion-safe:animate-dockBounce") }>
                <div className="w-7 h-7 text-zinc-800 dark:text-zinc-100">{app.icon}</div>
              </div>
              <div className={clsx("absolute -bottom-2 h-1 w-1 rounded-full", isOpen && !isMinimized ? "bg-accent" : "bg-transparent")} />
            </button>
          );
        })}
      </div>
    </div>
  );
}

function AboutIcon() {
  return (
    <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
      <path d="M12 2a10 10 0 1 0 0 20 10 10 0 0 0 0-20zm.75 5.5a.75.75 0 1 1-1.5 0 .75.75 0 0 1 1.5 0zM11 10.25a1 1 0 0 1 2 0v6a1 1 0 1 1-2 0v-6z" />
    </svg>
  );
}
function PythonIcon(props: React.SVGProps<SVGSVGElement>) {
  return (
    <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true" role="img" aria-label="Python Editor" {...props}>
      {/* Simple code-brackets icon to represent the editor */}
      <path d="M7.5 5.5a1 1 0 0 1 .7 1.7L5.4 10l2.8 2.8a1 1 0 1 1-1.4 1.4L3.3 10.7a1.99 1.99 0 0 1 0-2.8L6.8 4.4a1 1 0 0 1 .7-.3zM16.5 5.5a1 1 0 0 0-.7 1.7l2.8 2.8-2.8 2.8a1 1 0 1 0 1.4 1.4l3.5-3.5c.78-.78.78-2.05 0-2.83L17.2 4.4a1 1 0 0 0-.7-.3zM12 16.5c-.6 0-1 .4-1 1v1a1 1 0 1 0 2 0v-1c0-.6-.4-1-1-1z" />
    </svg>
  );
}
function ProjectsIcon() {
  return (
    <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
      <path d="M4 5a3 3 0 0 1 3-3h10a3 3 0 0 1 3 3v4H4V5zm0 6h16v8a3 3 0 0 1-3 3H7a3 3 0 0 1-3-3v-8z" />
    </svg>
  );
}

function DiscordIcon(props: React.SVGProps<SVGSVGElement>) {
  return (
    <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true" role="img" aria-label="Discord icon" {...props}>
      <path d="M20.317 4.369A19.791 19.791 0 0 0 16.558 3c-.204.368-.44.864-.603 1.25a18.27 18.27 0 0 0-3.91 0A7.04 7.04 0 0 0 11.442 3c-1.35.247-2.648.68-3.759 1.369C4.684 7.273 4 10.07 4 12.813c0 .077.002.154.006.23 1.45 1.077 2.84 1.734 4.163 2.104.33-.456.624-.94.878-1.448-1.383-.42-2.044-1.065-2.044-1.065s.115.082.325.197c.01.006.02.011.03.017.027.016.055.03.084.046.239.13.477.246.705.346.397.172.872.347 1.418.482.722.181 1.55.326 2.472.326s1.75-.145 2.472-.326c.546-.135 1.021-.31 1.418-.482.24-.104.488-.223.74-.36.007-.004.014-.007.02-.011.227-.121.385-.221.385-.221s-.672.655-2.06 1.074c.252.506.545.988.873 1.443 1.33-.372 2.726-1.03 4.183-2.113.004-.08.006-.161.006-.241 0-2.742-.684-5.539-3.126-8.444zM9.498 12.687c-.683 0-1.238-.6-1.238-1.34 0-.739.551-1.34 1.238-1.34.694 0 1.245.601 1.238 1.34 0 .74-.551 1.34-1.238 1.34zm5.004 0c-.683 0-1.238-.6-1.238-1.34 0-.739.551-1.34 1.238-1.34.694 0 1.245.601 1.238 1.34 0 .74-.544 1.34-1.238 1.34z"/>
    </svg>
  );
}


function SkillsIcon() {
  return (
    <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
      <path d="M12 2l2.5 5L20 8l-4 4 .9 6L12 16l-4.9 2L8 12 4 8l5.5-1L12 2z" />
    </svg>
  );
}

function MailIcon() {
  return (
    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="1.5" aria-hidden="true">
      <rect x="3" y="5" width="18" height="14" rx="2" />
      <path d="M3 7l9 6 9-6" />
    </svg>
  );
}
