"use client";
import { useEffect, useRef, useState } from "react";
import { AppId, useWindowStore } from "@/lib/windowStore";
import clsx from "clsx";

export default function AppWindow({ id, children }: { id: AppId; children: React.ReactNode }) {
  const win = useWindowStore((s) => s.windows[id]);
  const zTop = useWindowStore((s) => s.zTop);
  const focus = useWindowStore((s) => s.focus);
  const minimize = useWindowStore((s) => s.minimize);
  const close = useWindowStore((s) => s.close);
  const move = useWindowStore((s) => s.move);
  const resize = useWindowStore((s) => s.resize);
  const toggleMaximize = useWindowStore((s) => s.toggleMaximize);

  const dragging = useRef(false);
  const [offset, setOffset] = useState({ x: 0, y: 0 });
  const [mounted, setMounted] = useState(false);

  useEffect(() => setMounted(true), []);

  if (!win || win.minimized) return null;

  const onMouseDownHeader = (e: React.MouseEvent) => {
    dragging.current = true;
    setOffset({ x: e.clientX - win.x, y: e.clientY - win.y });
    focus(id);
  };
  const onMouseMove = (e: React.MouseEvent) => {
    if (!dragging.current) return;
    move(id, { x: e.clientX - offset.x, y: e.clientY - offset.y });
  };
  const onMouseUp = () => (dragging.current = false);

  const isFocused = win.z === zTop;

  return (
    <div
      role="dialog"
      aria-label={win.title}
      style={{ left: win.x, top: win.y, width: win.w, height: win.h, zIndex: win.z }}
      className={clsx(
        "fixed rounded-2xl border overflow-hidden select-none bg-white dark:bg-zinc-900",
        // elevation and border polish on focus
        isFocused ? "shadow-2xl border-black/10 dark:border-white/10" : "shadow-xl border-black/5 dark:border-white/5",
        // smooth fade/slide-in on mount, but avoid animating position during drag
        mounted ? "opacity-100 translate-y-0" : "opacity-0 translate-y-2",
        "transition-[opacity,box-shadow,filter] duration-300 ease-out motion-reduce:transition-none motion-reduce:transform-none"
      )}
      onMouseMove={onMouseMove}
      onMouseUp={onMouseUp}
      onMouseLeave={onMouseUp}
      onMouseDown={() => focus(id)}
    >
      <div
        className={clsx(
          "glass h-7 flex items-center gap-2 px-3 text-xs",
          "border-b border-black/5 dark:border-white/5 cursor-move"
        )}
        onMouseDown={onMouseDownHeader}
        onDoubleClick={() => toggleMaximize(id)}
      >
        <div className="flex items-center gap-2">
          <button
            aria-label="Close"
            className="h-3 w-3 rounded-full bg-[#ff5f57]"
            onMouseDown={(e) => e.stopPropagation()}
            onClick={(e) => {
              e.stopPropagation();
              close(id);
            }}
          />
          <button
            aria-label="Minimize"
            className="h-3 w-3 rounded-full bg-[#febc2e]"
            onMouseDown={(e) => e.stopPropagation()}
            onClick={(e) => {
              e.stopPropagation();
              minimize(id);
            }}
          />
          <button
            aria-label="Zoom"
            className="h-3 w-3 rounded-full bg-[#28c840]"
            onMouseDown={(e) => e.stopPropagation()}
            onClick={(e) => {
              e.stopPropagation();
              focus(id);
              toggleMaximize(id);
            }}
          />
        </div>
        <div className="mx-2 font-medium truncate">{win.title}</div>
      </div>
      <div className="h-[calc(100%-28px)] overflow-auto">
        {children}
      </div>
    </div>
  );
}
