"use client";
import * as React from "react";
import MenuBar from "@/components/MenuBar";
import Dock from "@/components/Dock";
import AppWindow from "@/components/AppWindow";
import { useWindowStore } from "@/lib/windowStore";
import AboutApp from "@/components/apps/About";
// import NotesApp from "@/components/apps/Notes";
import PythonEditor from "@/components/apps/PythonEditor";
import ProjectsApp from "@/components/apps/Projects";
import SkillsApp from "@/components/apps/Skills";
import ContactApp from "@/components/apps/Contact";
import BackgroundFX from "@/components/BackgroundFX";

export default function Page() {
  return (
    <main className="relative min-h-screen">
      {/* Top menu bar */}
      <MenuBar />

      {/* Animated background */}
      <BackgroundFX />

      {/* Desktop area - could add desktop icons later */}
      <div className="pt-8 pb-24"></div>

      {/* Windows */}
      <WindowsLayer />

      {/* Dock */}
      <Dock />
    </main>
  );
}

function WindowsLayer() {
  // Separate client component to read state
  return <WindowsClient />;
}

function WindowsClient() {
  const windows = useWindowStore((s) => s.windows);
  return (
    <>
      {windows.about && (
        <AppWindow id="about">
          <AboutApp />
        </AppWindow>
      )}
      {windows.python && (
        <AppWindow id="python">
          <PythonEditor />
        </AppWindow>
      )}
      {windows.projects && (
        <AppWindow id="projects">
          <ProjectsApp />
        </AppWindow>
      )}
      {windows.presence && (
        <AppWindow id="presence">
          <PresenceApp />
        </AppWindow>
      )}
      {windows.skills && (
        <AppWindow id="skills">
          <SkillsApp />
        </AppWindow>
      )}
      {windows.contact && (
        <AppWindow id="contact">
          <ContactApp />
        </AppWindow>
      )}
    </>
  );
}

// Inline Lanyard hook & Presence app to avoid extra files
function useLanyard(userId: string, { interval = 15000 }: { interval?: number } = {}) {
  const [data, setData] = React.useState<any>(null);
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState<string | null>(null);
  const url = React.useMemo(() => `https://api.lanyard.rest/v1/users/${userId}`, [userId]);
  React.useEffect(() => {
    let active = true;
    let timer: number | undefined;
    let controller: AbortController | null = null;
    const fetchOnce = async () => {
      try {
        setError(null);
        controller?.abort();
        controller = new AbortController();
        const res = await fetch(url, { signal: controller.signal, cache: 'no-store' });
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const json = await res.json();
        if (!active) return;
        setData(json?.data ?? null);
        setLoading(false);
      } catch (e: any) {
        if (!active) return;
        if (e?.name === 'AbortError') return;
        setError(e?.message ?? 'Failed to load presence');
        setLoading(false);
      }
    };
    setLoading(true);
    fetchOnce();
    timer = window.setInterval(fetchOnce, Math.max(5000, interval));
    return () => {
      active = false;
      if (timer) window.clearInterval(timer);
      controller?.abort();
    };
  }, [url, interval]);
  return { data, loading, error } as const;
}

function statusToColor(status: string): string {
  switch (status) {
    case 'online': return 'bg-emerald-500';
    case 'idle': return 'bg-amber-400';
    case 'dnd': return 'bg-rose-500';
    default: return 'bg-zinc-400';
  }
}

function avatarUrl(user: any, size: number = 128): string {
  if (user?.avatar) return `https://cdn.discordapp.com/avatars/${user.id}/${user.avatar}.png?size=${size}`;
  const disc = user?.discriminator && user.discriminator !== '0' ? Number(user.discriminator) % 5 : 0;
  return `https://cdn.discordapp.com/embed/avatars/${disc}.png?size=${size}`;
}

function Card({ title, children }: { title: string; children: React.ReactNode }) {
  return (
    <div className="rounded-xl border border-black/10 dark:border-white/10 bg-white/70 dark:bg-zinc-900/70">
      <div className="px-3 py-2 border-b border-black/5 dark:border-white/5 text-xs text-zinc-500">{title}</div>
      <div className="p-3">{children}</div>
    </div>
  );
}

function PresenceApp() {
  const { data, loading, error } = useLanyard('1036309466125176884', { interval: 15000 });
  const primary = React.useMemo(() => data?.activities?.find((a: any) => a.type === 0) || data?.activities?.[0], [data]);
  return (
    <div className="p-4 text-sm text-zinc-800 dark:text-zinc-100 grid gap-4">
      <div className="flex items-center gap-3">
        <div className="relative">
          <img src={data ? avatarUrl(data.discord_user, 128) : 'https://cdn.discordapp.com/embed/avatars/0.png?size=128'} alt="avatar" className="size-14 rounded-2xl object-cover border border-black/10 dark:border-white/10" />
          {data && <span className={`absolute -bottom-1 -right-1 size-3 rounded-full ring-2 ring-white dark:ring-zinc-900 ${statusToColor(data.discord_status)}`} />}
        </div>
        <div>
          <div className="text-base font-semibold tracking-tight">{loading ? 'Loading…' : data?.discord_user?.username ?? 'Unknown'}</div>
          <div className="text-xs text-zinc-500">{error ? 'Offline' : data?.discord_status ?? 'offline'}</div>
        </div>
      </div>
      <div className="grid gap-3 md:grid-cols-2">
        <Card title="Status">
          {loading ? <div className="h-4 rounded bg-black/5 dark:bg-white/10 animate-pulse" /> : (
            <div className="text-xs grid gap-1">
              <div className="flex items-center gap-2">
                <span className={`inline-block size-2 rounded-full ${statusToColor(data?.discord_status)}`} />
                <span className="capitalize">{data?.discord_status}</span>
              </div>
              <div className="text-zinc-500">User ID: {data?.discord_user?.id}</div>
            </div>
          )}
        </Card>
        <Card title="Spotify">
          {loading ? <div className="h-4 rounded bg-black/5 dark:bg-white/10 animate-pulse" /> : !data?.spotify ? (
            <div className="text-xs text-zinc-500">Not listening</div>
          ) : (
            <div className="flex items-center gap-3">
              <img src={data.spotify.album_art_url} alt="album" className="size-12 rounded-md border border-black/10 dark:border-white/10" />
              <div className="min-w-0">
                <div className="text-sm font-medium truncate">{data.spotify.song}</div>
                <div className="text-xs text-zinc-500 truncate">{data.spotify.artist} — {data.spotify.album}</div>
                <Progress start={data.spotify.timestamps.start} end={data.spotify.timestamps.end} />
              </div>
            </div>
          )}
        </Card>
      </div>
      <Card title="Activities">
        {loading ? (
          <div className="grid gap-2">
            <div className="h-4 rounded bg-black/5 dark:bg-white/10 animate-pulse" />
            <div className="h-4 rounded bg-black/5 dark:bg-white/10 animate-pulse" />
            <div className="h-4 rounded bg-black/5 dark:bg-white/10 animate-pulse" />
          </div>
        ) : (
          (data?.activities?.length ? (
            <ul className="grid gap-2">
              {data.activities.map((a: any, i: number) => (
                <li key={`${a.id ?? a.name}-${i}`} className="flex items-start gap-3">
                  <div className="size-8 rounded-lg bg-black/5 dark:bg-white/10 grid place-items-center text-[10px] uppercase tracking-wide">
                    {kind(a.type)}
                  </div>
                  <div className="min-w-0">
                    <div className="text-sm font-medium truncate">{a.name}</div>
                    {(a.details || a.state) && <div className="text-xs text-zinc-500 truncate">{a.details || a.state}</div>}
                  </div>
                </li>
              ))}
            </ul>
          ) : <div className="text-xs text-zinc-500">No recent activities</div>)
        )}
      </Card>
    </div>
  );
}

function Progress({ start, end }: { start: number; end: number }) {
  const pct = Math.min(100, Math.max(0, ((Date.now() - start) / (end - start)) * 100));
  return (
    <div className="h-1 bg-black/10 dark:bg-white/10 rounded mt-1">
      <div className="h-1 bg-accent rounded" style={{ width: `${pct}%` }} />
    </div>
  );
}

function kind(t: number) {
  switch (t) {
    case 0: return 'play';
    case 2: return 'listen';
    case 3: return 'watch';
    case 5: return 'comp';
    default: return 'act';
  }
}
